//*****************************************************************************
//
// File:             progparams.cpp
//
// Purpose:          Implementation of the progparams class for rmsps.
//
// Author:           Michael Edwards - m@michael-edwards.org
//
// Date:             September 19th 2001
//
// License:          Copyright (C) 2001 Michael Edwards
//
//                   This file is part of rmsps.  
//
//                   rmsps is free software; you can redistribute it and/or
//                   modify it under the terms of the GNU General Public
//                   License as published by the Free Software Foundation;
//                   either version 2 of the License, or (at your option) any
//                   later version.
//
//                   rmsps is distributed in the hope that it will be useful,
//                   but WITHOUT ANY WARRANTY; without even the implied
//                   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//                   PURPOSE.  See the GNU General Public License for more
//                   details.
//
//                   You should have received a copy of the GNU General Public
//                   License along with rmsps; if not, write to the Free
//                   Software Foundation, Inc., 59 Temple Place, Suite 330,
//                   Boston, MA 02111-1307 USA
//
// $$ Last modified: 19:43:32 Thu Mar 14 2002 CET
//
//*****************************************************************************

#include "progparams.h"

//*****************************************************************************

progparams::progparams(void) 
{
    Width = 0;
    Height = 0;
    XScale = 0.0;
    RMSSize = 0;
    DefaultRMSSize = 10.0;
    RMSScaler = 1.0;
    strcpy(Shading, "0.0 setgray");
    HLines = 8;
    OutDir[0] = '\0';
    strcpy(Font, "Times-Roman");
    FontSize = 8;
    YOffset = FontSize + 2;
    Units = CM;
    ReadOffset = 0.0;
    WriteOffset = 0.0;
    LineWidth = 0.2; // points
    TimeLine = 1;
}

//*****************************************************************************

int progparams::parse(char* line)
{
    // None of these names can contain another name as a substring otherwise
    // the wrong parameter case will be triggered.  i.e. "line-width" would
    // trigger "width" in the rather simple but primitive parsing algorithm I
    // have here.
    static const char* const parameters[] = 
        { "width", "height", "xscale", "rms-size", "rms-scaler", "sndfile",
          "gray", "outdir", "hlines", "colour", "font", "font-size", "units",
          "read-offset", "write-offset", "line-thickness", "time-line" };
    int num = sizeof parameters / sizeof parameters[0];
    char* p;
    char* ptr;
    char buf[512];
    char linedown[1024];
    int got = 0;
    double tmp;
    double red;
    double green;
    double blue;
    static const char sunits[][4] = { "in", "cm", "pt" };

    // This line had to be < 1024 chars long, right?  If not, at least no
    // crash... 
    strncpy(linedown, line, sizeof linedown);
    // Just to make sure...
    linedown[(sizeof linedown) - 1] = '\0';
    stringdowncase(linedown);

    // We look for each of the parameters in order given above, which means if
    // gray comes after color on the same line, the color will prevail because
    // it gets set after gray.  This is a bug but acceptable because we
    // shouldn't set gray and color at the same time anyway.  Fix this later
    // when necessary?
    for (int i = 0; i < num; ++i) {
        p = (char*)parameters[i];
        if ((ptr = strstr(linedown, p))) {
            ptr += strlen(p);
            while (isspace(*ptr))
                ++ptr;
            if (*ptr == '=') {
                char j = 0;
                ++ptr;
                while (isspace(*ptr))
                    ++ptr;
                // we've operated on the string-downcased line up to now; 
                // now switch to the original;
                int index = ptr - linedown;
                ptr = line + index;
                while (!isspace(*ptr) && *ptr !='\0')
                  buf[j++] = *ptr++;
                buf[j] = '\0';
                if (!strlen(buf))
                    error("progparams::parse",
                          "Got '=' but no parameter value:\n%s\n\n", line);
                ++got;
            }
            else continue;
            // we've got the parameter, now use it
            switch (i) {
            case 0:
                tmp = atof(buf);
                Width = InPoints(tmp);
                printf("Page width = %.3f%s (%d points)\n", tmp, 
                       sunits[Units], Width);
                break;
            case 1:
                tmp = atof(buf);
                Height = InPoints(tmp);
                printf("Page height = %.3f%s (%d points)\n", tmp, 
                       sunits[Units], Height);
                break;
            case 2:
                XScale = atof(buf);
                printf("X scale max = %.3f\n", XScale);
                break;
            case 3:
                if (!SoundFile.isopen())
                    error("progparams::parse",
                          "You must specify a sound file before "
                          "setting the RMS size.");
                tmp = atof(buf);
                RMSSize = GetRMSSize(tmp);
                printf("RMS size = %.3f millisecs (%d samples)\n",
                       tmp, RMSSize);
                break;
            case 4:
                RMSScaler = atof(buf);
                printf("RMS Y scaler = %.3f\n", RMSScaler);
                break;
            case 5:
                SoundFile.close();
                SoundFile.open(buf);
                printf("Input sound file = '%s'\n", buf);
                break;
            case 6:
                tmp = atof(buf);
                sprintf(Shading, "%.3f setgray", tmp);
                printf("Gray fill = %.3f\n", tmp);
                break;
            case 7:
                strcpy(OutDir, buf);
                finalslash(OutDir);
                printf("Output directory = '%s'\n", OutDir);
                break;
            case 8:
                HLines = atoi(buf);
                printf("Horizontal Rule Lines = %i\n", HLines);
                break;
            case 9:
                if (sscanf(buf, "%lf,%lf,%lf", &red, &green, &blue) != 3)
                    error("progparams::parse",
                          "RGB values must be three values separated by "
                          "commas but no space: %s\n", buf);
                sprintf(Shading, "%.3f %.3f %.3f setrgbcolor", 
                        red, green, blue);
                printf("Colour fill = Red: %.3f, Green: %.3f, Blue: %.3f\n",
                       red, green, blue);
                break;
            case 10:
                strcpy(Font, buf);
                printf("Font = '%s'\n", Font);
                break;
            case 11:
                FontSize = atoi(buf);
                YOffset = FontSize + 2;
                printf("Font size = %d\n", FontSize);
                break;
            case 12:
                stringdowncase(buf);
                if (!strcmp(buf, "cm")) {
                    Units = CM;
                    printf("Units = centimeters\n");
                }
                else if (!strcmp(buf, "in")) {
                    Units = IN;
                    printf("Units = inches\n");
                }
                else if (!strcmp(buf, "pt")) {
                    Units = PT;
                    printf("Units = points\n");
                }
                else error("progparams::parse", 
                           "Illegal units: %s.  Legal values are "
                           "'in' 'cm' and 'pt'", buf);
                break;
            case 13:
                ReadOffset = getsecs(buf);
                printf("Read Offset = %f seconds\n", ReadOffset);
                break;
            case 14:
                WriteOffset = getsecs(buf);
                printf("Write Offset = %f seconds\n", WriteOffset);
                break;
            case 15:
                LineWidth = atof(buf);
                printf("Postscript Line Thickness = %f points\n", LineWidth);
                break;
            case 16:
                TimeLine = strcmp(stringdowncase(buf), "no");
                printf("TimeLine = %s\n", TimeLine ? "yes" : "no");
                break;
            }
        }
    }
    return got;
}


//*****************************************************************************

void progparams::CheckRequired(void)
{
    if (!SoundFile.isopen())
        missingparam("sndfile");
    if (!Width)
        missingparam("width");
    if (!Height)
        missingparam("height");
    if (!XScale)
        missingparam("xscale");
    if (!RMSSize)
        // default rms size is 10ms
        RMSSize = GetRMSSize(DefaultRMSSize);
}

//*****************************************************************************

// Returns x Units in points--rounded up!  No fractional points!

int progparams::InPoints(double x)
{
    double mult = -1.0;

    switch (Units) {
    case CM:
        mult = 72.0 / 2.54;
        break;
    case IN:
        mult = 72.0;
        break;
    case PT:
        mult = 1.0;
        break;
    }
    return int(ceil(x * mult));
}

//*****************************************************************************

// The number of samples needed from the sound file to give ms millisecs.

int progparams::GetRMSSize(double ms)
{
    if (!SoundFile. isopen())
        error("progparams::GetRMSSize", 
              "Sound file must be open before calling!");
    return int(ms * 0.001 * SoundFile.GetChannels() * SoundFile.GetSrate());
}

//*****************************************************************************
//*****************************************************************************

// Make a string all lower case. N.B.  This is a **destructive** routine.

char* stringdowncase(char* string)
{
    char* c;

    for (c = string; *c != '\0'; ++c) {
        *c = tolower(*c);
    }
    return string;
}

//*****************************************************************************

// Watch out!  Changes it's argument and doesn't check boundaries when
// inserting an extra char! 

void finalslash(char* path)
{
#ifdef WIN32
    char dirdiv = '\\';
#else
    char dirdiv = '/';
#endif
    int len = strlen(path);
    
    if (path[len - 1] != dirdiv) {
        path[len] = dirdiv;
        path[len + 1] = '\0';
    }
}

//*****************************************************************************

// EOF progparams.cpp
