//*****************************************************************************
//
// File:             rmsps.cpp
//
// Purpose:          Implementation of an RMS (loudness) sound file analysis
//                   program that writes Postscript files of the RMS curve of a
//                   sound file for inclusion/creation of scores.  
//
// Author:           Michael Edwards - m@michael-edwards.org
//
// Date:             August 2nd 2001
//
// License:          Copyright (C) 2001 Michael Edwards
//
//                   This file is part of rmsps.  
//
//                   rmsps is free software; you can redistribute it and/or
//                   modify it under the terms of the GNU General Public
//                   License as published by the Free Software Foundation;
//                   either version 2 of the License, or (at your option) any
//                   later version.
//
//                   rmsps is distributed in the hope that it will be useful,
//                   but WITHOUT ANY WARRANTY; without even the implied
//                   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//                   PURPOSE.  See the GNU General Public License for more
//                   details.
//
//                   You should have received a copy of the GNU General Public
//                   License along with rmsps; if not, write to the Free
//                   Software Foundation, Inc., 59 Temple Place, Suite 330,
//                   Boston, MA 02111-1307 USA
//
// $$ Last modified: 11:54:42 Sat Feb 14 2004 GMT Standard Time
//
//*****************************************************************************

#include "rmsps.h"

static const int MAXLINE = 1024;
static const int MAXSEGS = 1024;
static const char* const VERSION = "1.03";
// points needed to write a time as MM:SS.sss (8 pt courier)
static const int TIME_WIDTH = 45; 

int main(int argc, char** argv)
{
    char buf[MAXLINE];
    double times[MAXSEGS];
    double xs[MAXSEGS];
    char epsfile[512];
    char fullpath[512];
    int pairs;
    FILE* fp;
    int warnings = 0;
    int files = 0;
    progparams pps;

    printf("\n    rmsps version %s (%s %s)\n"
           "    Copyright (C) 2001 - Michael Edwards - m@michael-edwards.org\n"
           "    rmsps comes with ABSOLUTELY NO WARRANTY; for details see \n"
           "    COPYING.txt which came with the program.\n"
           "    This is free software, and you are welcome to redistribute\n"
           "    it under certain conditions; see COPYING.txt for details.\n\n",
           VERSION, __TIME__, __DATE__);

    if (argc != 2)
        error("main", "Usage: rmsps input-file\n\n");
    if ((fp = fopen(argv[1], "r")) == NULL)
        error("main", "Can't open \"%s\".\nUsage: rmsps input-file\n\n",
              argv[1]);

    while (getline(fp, buf)) {
        if (pps.parse(buf)) {
            // not strictly necessary but clear...
            continue;
        }
        else {
            pairs = GetPageParameters(buf, epsfile, times, xs);
            pps.CheckRequired();
            strcpy(fullpath, pps.GetOutDir());
            strcat(fullpath, epsfile);
            ++files;
            warnings +=
                writeepsfile(fullpath, times, xs, pairs - 1, &pps);
        }
    }
    printf("\nWrote %d files.  %d warnings.\n\n", files, warnings);
    return 0;
}

//*****************************************************************************

char* secs_to_mins(double secs)
{
    static char buf[128];
    int mins = int(secs / 60.0);

    if (secs < 60.0)
      sprintf(buf, "%06.3f", secs);
    else sprintf(buf, "%d:%06.3f", mins, secs - (60.0 * mins));
    char* ptr = buf;
    ptr += strlen(ptr) - 1;
    while (*ptr == '0')
        --ptr;
    if (*ptr == '.')
        *ptr = '\0';
    else *++ptr = '\0';
    ptr = buf;
    if (*ptr == '0')
        ptr++;
    return ptr;
}

//*****************************************************************************

// Print the horizontal dashed lines across the image.
// height and width are in points

void verticalscale(FILE* eps, int num, int height, int width, 
                   const char* shading, int yoffset)
{
    // draw the vertical line that starts this image on the left?
    // vline(eps, 0, 0, double(height));
    double incy = (double(height) - double(yoffset)) / double(num);
    double y = yoffset + incy;

    // draw the base line
    hline(eps, 0, yoffset, width);

    // uncomment this and the final line if the broken lines should be the same
    // colour as the wave form fill.
    // fprintf(eps, "\n\n%%BEGIN VERTICAL SCALE\n\n%s\n\n", shading);
    for (int i = 0; i < num; ++i) {
        brokenhline(eps, 0, y, width);
        y += incy;
    }
    // fprintf(eps, "\n\n%%END VERTICAL SCALE\n\n0.0 setgray\n\n");
}

//*****************************************************************************

inline void line(FILE* epsfile, double x1, double y1, double x2, double y2)
{
    fprintf(epsfile, 
            "%.3f %.3f %.3f %.3f LS\n",
            x1, y1, x2, y2);
}

//*****************************************************************************

inline void hline(FILE* epsfile, double x1, double y, double x2)
{
    line(epsfile, x1, y, x2, y);
}

//*****************************************************************************

inline void vline(FILE* epsfile, double x, double y1, double y2)
{
    line(epsfile, x, y1, x, y2);
}

//*****************************************************************************

void brokenhline(FILE* epsfile, double x1, double y1, double x2)
{
    double len = 2.0;
    double x = x1;
    double dist = (x2 - x1);
    int num = int((dist / len) * 0.5);

    // we need an odd number of lines to reach our goal
    if (!(num & 1))
        ++num;
    double total_space = dist - (len * num);
    double space = total_space / (num - 1);
    double xinc = len + space;
    for (int i = 0; i < num; ++i) {
        hline(epsfile, x, y1, x + len); 
        x += xinc;
    }
}

//*****************************************************************************

void ruler(double start_secs, double end_secs, double xstart, double xend,
           int fontsize, int yoffset, FILE* epsfile)
{
    double total_secs = end_secs - start_secs;
    double total_x = xend - xstart;
    double xpersec = total_x / total_secs;
    static const double steps[] = 
        { 0.01, 0.1, 0.2, 0.5, 1.0, 2.0, 5.0, 10.0, 30.0, 60.0, 120.0, 300.0 };
    static const int numsteps = sizeof steps / sizeof steps[0];
    double incsecs = 0.0;
    double time_width = double(TIME_WIDTH) * (double(fontsize) / 8.0);

    for (int i = 0; i < numsteps; ++i) {
        if (xpersec >= (time_width / steps[i])) {
            incsecs = steps[i];
            break;
        }
    }
    if (!incsecs) {
        printf("Increment too small or too large.  Not writing ruler.\n");
        return; // don't bother....
    }

    double incx = incsecs * xpersec;
    double xmin = xstart + time_width;
    double secs = incsecs * double(1 + int(start_secs / incsecs));
    double xoffset = xstart + (xpersec * (secs - start_secs));
    
    while (xoffset < xmin) {
        xoffset += incx;
        secs += incsecs;
    }
    fprintf(epsfile, "\n\n%%START RULER\n");
    rulerwritetime(epsfile, xstart, yoffset, start_secs, 5.0);
    while (xoffset <= (xend - time_width)) {
        rulerwritetime(epsfile, xoffset, yoffset, secs, 5.0);
        xoffset += incx;
        secs += incsecs;
    }
    fprintf(epsfile, "%%END RULER\n\n");
}

//*****************************************************************************

void rulerwritetime(FILE* epsfile, double x, double y, double secs, 
                    double notch_depth)
{
    vline(epsfile, x, y - notch_depth, y);
    fprintf(epsfile, "%.3f 0 N (%s) S\n",
            x, secs_to_mins(secs));
}

//*****************************************************************************

// returns the amount of warnings issued.

int writeepsfile(char* epsfile, double* times, double* xs, int num_segments,
                 progparams* pps)
{
    FILE* fp;
    int seg;
    double start_secs;
    double start_secs_written;
    double end_secs;
    double end_secs_written;
    int do_samples;
    double xdiff;
    double xamount;
    double xinc;
    double xstart;
    double xsegpixels;
    int num_rms_calls;
    int ps_image_width_pts = pps->GetWidth();
    int ps_image_height_pts = pps->GetHeight();
    int yoffset = pps->GetYOffset();
    double yscaler = (double(ps_image_height_pts) - yoffset) *
        pps->GetRMSScaler();
    time_t t = time(NULL);
    int warnings = 0;
    sndfile* sf = pps->GetSoundFile();

    if ((fp = fopen(epsfile, "w")) == NULL)
        error("writeepsfile", "Can't open \"%s\" for writing.", epsfile);
    fprintf(fp, "%%!PS-Adobe EPSF-3.0\n"
            "%%%%Creator rmsps %s (%s %s)\n"
            "%%%%Author Michael Edwards - m@michael-edwards.org\n"
            "%%%%CreationDate %s"
            "%%%%BoundingBox: 0 0 %d %d\n"
            // definition of line function.
            "/LS {newpath moveto lineto closepath stroke} bind def\n"
            "/L {lineto} bind def\n"
            "/N {newpath moveto} bind def\n"
            "/S {show} bind def\n"
            "%%%%EndComments\n%%%%BeginProlog\n%%%%EndProlog\n"
            "%%%%BeginSetup\n%%%%EndSetup\n"
            "/%s findfont\n%d scalefont\nsetfont\n"
            "%.3f setlinewidth\n\n",
            VERSION, __TIME__, __DATE__, ctime(&t), 
            ps_image_width_pts, ps_image_height_pts, pps->GetFont(),
            pps->GetFontSize(), pps->GetLineWidth());
    // put a rectangle around the image??
    // drawrect(fp, 0, 0, ps_image_width_pts, ps_image_height_pts);
    verticalscale(fp, pps->GetHLines(), ps_image_height_pts,
                  ps_image_width_pts, pps->GetShading(), yoffset);
    // the horizontal line at the base (0) of the RMS curve?
    // hline(fp, 0, yoffset, ps_image_width_pts);

    printf("\nWriting '%s'\n", epsfile);

    for (seg = 0; seg < num_segments; ++seg) {
        start_secs = times[seg];
        start_secs_written = start_secs + pps->GetWriteOffset();
        start_secs += pps->GetReadOffset();
        end_secs = times[seg + 1];
        end_secs_written = end_secs + pps->GetWriteOffset();
        end_secs += pps->GetReadOffset();
        if (start_secs > end_secs) 
            error("writeepsfile", "start (%.3f) > end (%.3f)", 
                  start_secs, end_secs);
        printf("Seconds %.3f -> %.3f: ", start_secs, end_secs);
        do_samples = int((end_secs - start_secs) * sf->GetSrate() 
                        * sf->GetChannels());
        num_rms_calls = do_samples / pps->GetRMSSize();
        sf->seek(start_secs);
        xstart = (xs[seg] / pps->GetXScale()) * double(ps_image_width_pts);
        xdiff = xs[seg + 1] - xs[seg];
        // proportion of whole x axis this segment will take
        xamount = xdiff / pps->GetXScale();
        xsegpixels = xamount * double(ps_image_width_pts);
        xinc = xsegpixels / (double(num_rms_calls) - 1.0);
        if (pps->GetTimeLine()) 
            ruler(start_secs_written, end_secs_written, xstart, 
                  xstart + xsegpixels, pps->GetFontSize(), yoffset, fp);
        warnings += 
            writesegment(fp, num_rms_calls, xstart, xinc, yscaler,
                         // writeoffset and readoffset are independant of each
                         // other!
                         start_secs_written, pps);
    }
    fprintf(fp, "\n\nshowpage\n\n%%Trailer\n%%DocumentFonts:\n\n");
    fclose(fp);
    return warnings;
}

//*****************************************************************************

int writesegment(FILE* eps, int num_rms_calls, double xstart, double xinc, 
                 double yscaler, double starttime, progparams* pps)
{
    double rms = 0.0;
    double rms_points;
    int i;
    double x = xstart;
    double max_rms = 0.0;
    int warnings = 0;
    int image_ht_pts = pps->GetHeight();
    int yoffset = pps->GetYOffset();

    // do this silly thing so the SGI compiler doesn't complain...
    starttime = starttime;
#ifdef SHOW_SEGMENTS
    // draw the vertical line that starts this segment
    vline(eps, xstart, yoffset - 5.0, 2 * yscaler);
    // write the start time of this segment
    fprintf(eps, "%.3f 0 N (%s) S\n",
            xstart, secs_to_mins(starttime));
#endif
    
    // start the rms drawing
    fprintf(eps, "%.3f %d N\n", xstart, yoffset);
    for (i = 0; i < num_rms_calls; ++i) {
        rms = getrms(pps->GetSoundFile(), pps->GetRMSSize());
        // exponentiation added 16/12/03
        rms = pow(rms, pps->GetRMSExpt());
        rms_points = rms * yscaler;
        if (rms_points > max_rms)
            max_rms = rms_points;
        fprintf(eps, "%.3f %.3f L\n", x, rms_points + yoffset);
        x += xinc;
    }
    max_rms += yoffset;
    if (max_rms > double(image_ht_pts)) {
        printf("\n*** Warning: Maximum (scaled) RMS (%.3f points) > "
               "Image Height (%d points).\n"
               "Reduced rms-scaler recommended.\n",
               max_rms, image_ht_pts);
        ++warnings;
    }
    else printf("Max RMS: %.3f = %.3f / %d points\n", 
                rms, max_rms, image_ht_pts);
    x -= xinc;
    fprintf(eps, "%.3f %d L %.3f %d L closepath\n\n",
            x, yoffset, xstart, yoffset);
    fprintf(eps, "%s\nfill\n0.0 setgray\n\n", pps->GetShading());
    return warnings;
}


//*****************************************************************************

double getrms(sndfile* sf, const int rms_size)
{
    int i;
    double sample;
    double samplesq;
    double sum = 0.0;
    double sumsq = 0.0;
    float* samples;

    samples = sf->GetSamples(rms_size);
    
    for (i = 0; i < rms_size; ++i) {
        // NB 16 bit assumed!!!
        sample = double(samples[i]);
        samplesq = sample * sample;
        sum += sample;
        sumsq += samplesq;
    }
    
    double mean = sum / double(rms_size);
    double meansq = sumsq / double(rms_size);
    double var = meansq - (mean * mean);
    double rms = sqrt(var);

    // make the values logarithmic to boost the lower amplitudes into
    // meaningful values.
    rms = rmslog(rms, 40.0);
    return rms;
}

//*****************************************************************************

double rmslog(double rms, double base)
{
    return mylog(1.0 + (rms * (base - 1.0)),
                 base);
}

//*****************************************************************************

inline double mylog(double x, double base)
{
    return log10(x) / log10(base);
}

//*****************************************************************************

// Not used but maybe later?  If so, replace the explicit ps commands with the
// defined ps functions.  
void drawrect(FILE* fp, int x1, int y1, int x2, int y2)
{
    fprintf(fp, "newpath %d %d moveto %d %d lineto %d %d lineto "
            "%d %d lineto %d %d lineto closepath stroke\n",
            x1, y1, x2, y1, x2, y2, x1, y2, x1, y1);
}

//*****************************************************************************

char* getline(FILE* fp, char* buf)
{
    char* ret;

    while ((ret = fgets(buf, MAXLINE, fp))) {
        stringtrim(buf);
        if (*buf == '#' || *buf == '\n' || *buf == '\0')
            continue;
        else break;
    }
    return ret;
}


//*****************************************************************************

int GetPageParameters(char* line, char* filename, double* times, double* xs)
{
    char* token = strtok(line, " ");
    int ti = 0;
    int xi = 0;

    strcpy(filename, token);
    int pairs;

    for (pairs = 0; token; ++pairs) {
        token = strtok(NULL, " ");
        if (token)
            xs[xi++] = atof(token);
        token = strtok(NULL, " ");
        if (token)
            times[ti++] = getsecs(token);
    }
    return --pairs;
}

//*****************************************************************************

// converts times in mins:secs format to seconds
// can't use strtok here because it's already in use....

double getsecs(char* time)
{
    char cmins[64] = "";
    char csecs[64] = "";
    double mins = 0.0;
    double secs;
    char* ptr;
    int got_colon = 0;
    int i = 0;
    
    for (ptr = time; *ptr != '\0'; ++ptr) {
        if (*ptr == ':') {
            got_colon = 1;
            cmins[i] = '\0';
            i = 0;
            continue;
        }
        if (got_colon)
            csecs[i++] = *ptr;
        else cmins[i++] = *ptr;
    }
    csecs[i] = '\0';

    secs = got_colon ? atof(csecs) : atof(cmins);
    mins = got_colon? atof(cmins) : 0.0;
    return (mins * 60.0) + secs;
}

//*****************************************************************************

// Trim leading and trailing white space from a string. N.B.  This is a
// **destructive** routine.

void stringtrim(char *str)
{
    char *ptr = str;

    // Return if empty string.
    if (*ptr == '\0')
        return;
    // Skip whitespace at beginning.
    while (*ptr != '\0' && isspace(*ptr++))
      ;
    // Copy the pointer back into <str>.
    strcpy(str, --ptr);
    // Go to the end of the string. 
    ptr += strlen(ptr);
    // Go backwards until we hit a non-whitespace character.
    while (isspace(*--ptr))
      ;
    *++ptr = '\0';
}

//*****************************************************************************

// EOF rmsps.cpp

