//*****************************************************************************
//
// File:             sndfile.cpp
//
// Purpose:          Open an AIFF, WAV or SND sound file and read it's header,
//                   storing sound info in appropriate class members.  Reads
//                   the different sound formats transparently, so all the user
//                   has to do is declare the class with the pathname as
//                   argument to the constructor and everything will be taken
//                   care of, including printing errors if the file isn't a
//                   sound etc.
//
// Author:           Michael Edwards - m@michael-edwards.org
//
// Date:             December 19th 1995
//
// License:          Copyright (C) 2001 Michael Edwards
//
//                   This file is part of rmsps.  
//
//                   rmsps is free software; you can redistribute it and/or
//                   modify it under the terms of the GNU General Public
//                   License as published by the Free Software Foundation;
//                   either version 2 of the License, or (at your option) any
//                   later version.
//
//                   rmsps is distributed in the hope that it will be useful,
//                   but WITHOUT ANY WARRANTY; without even the implied
//                   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//                   PURPOSE.  See the GNU General Public License for more
//                   details.
//
//                   You should have received a copy of the GNU General Public
//                   License along with rmsps; if not, write to the Free
//                   Software Foundation, Inc., 59 Temple Place, Suite 330,
//                   Boston, MA 02111-1307 USA
//
// $$ Last modified: 13:55:04 Fri Oct 12 2001 W. Europe Daylight Time
//
//*****************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <stdarg.h>
#include "sndfile.h"

//*****************************************************************************

// The number of characters to read from the header.  Should be easily enough
// to get all the info we need. 

static const int HDRSIZE = 4096;

//*****************************************************************************

// Constructor.  Needs the full path of the sound file (<path>) which is
// immediately stored in the class member <Path>. Open the file, allocate
// <Hdrbuf> and read in <HDRSIZE> bytes.  Test to see whether the file is
// AIFF or SND and process accordingly.  Free Hdrbuf at the end as we won't
// be needing header info anymore. 

sndfile::sndfile(const char *path)
{
    open(path);
    Samples = NULL;
    Swap = 0;
}
//*****************************************************************************

sndfile::sndfile(void)
{
    Samples = NULL;
    Swap = 0;
}

//*****************************************************************************

void sndfile::AllocSamples(int num)
{
    if (Samples == NULL || num > NumSamples) {
        Samples = new float[num];
        NumSamples = num;
    }
}

//*****************************************************************************

void sndfile::close(void)
{
    if (Fp)
        fclose(Fp);
    Fp = NULL;
}

//*****************************************************************************

void sndfile::seek(double secs)
{
    if (isopen()) {
        int start_byte = DataLocation
            // 16 bit assumed!!!!!
            + (2 * int(secs * double(Srate * Channels)));
        if (fseek(Fp, start_byte, SEEK_SET))
            error("sndfile::seek", "Can't seek to %f secs. (%d bytes)\n",
                  secs, start_byte);
        // make sure we get a new buffer of samples the next time GetSamples()
        // is called.  
        Index = BUFSIZ;
    }
    else error ("sndfile::seek", 
                "Sound file must be open before calling seek.");
}

//*****************************************************************************

float* sndfile::GetSamples(int num)
{
    static short samples[BUFSIZ];

    AllocSamples(num);
    for (int i = 0; i < num; ++i) {
        if (Index == BUFSIZ) {
            // 16 bit assumed!
            fread(samples, sizeof(short), BUFSIZ, Fp);
            if (ferror(Fp))
                error("sndfile::GetSamples", "Error reading sound file.\n");
            Index = 0;
            if (Swap)
                myswab((char*)samples, BUFSIZ * sizeof(short));
        }
        // NB 16 bit assumed!!!
        Samples[i] = float(samples[Index++]) / 32767.0f;
    }
    return Samples;
}

//*****************************************************************************

int sndfile::open(const char* path)
{
#ifdef WIN32
    char* mode = "rb";
#else
    char* mode = "r";
#endif

    Fp = fopen(path, mode);
    if (Fp == NULL)
        error("sndfile::sndfile", "Can't open '%s'.", path);
    Path = new char[strlen(path) + 1];
    strcpy(Path, path);
    Hdrbuf = new char[HDRSIZE];
    fread(Hdrbuf, 1, HDRSIZE, Fp);
    if (!strncmp(Hdrbuf, "FORM", 4))       // It's an AIFF file. 
        ReadAIFF(path);
    else if (!strncmp(Hdrbuf, ".snd", 4))  // It's an SND file. 
        ReadSND(path);
    else if (!strncmp(Hdrbuf, "RIFF", 4))  // It's a wave file.
        ReadWAV(path);
    else error("sndfile::sndfile",
               "Unsupported sound file type or file is not a sound file:\n"
               "\"%s\"",
               path);

    // Don't need the header info anymore. 
    delete[] Hdrbuf;
    Hdrbuf = NULL;
    Index = BUFSIZ;
    seek(0.0);
    return 0;
}

//*****************************************************************************

// Destructor.  Simply close the file pointer. 

sndfile::~sndfile(void)
{
    if (Fp)
        fclose(Fp);
}

//****************************************************************************

typedef struct s_waveheader {
    char riff[4];      // 0 "RIFF"
    long flength;      // 4 length of whole file in bytes - 8
    char wave[4];      // 8 "WAVE"
    char fmt[4];       // 12 "fmt "
    long format;       // 16 Length of fmt data (16 bytes)
    short type;        // 20 1 = PCM etc.
    short numchans;    // 22 1 = mono, 2 = stereo
    long srate;        // 24 samples per second
    long bps;          // 28 bytes per second = sample rate * block align
    short numchans2;   // 32 block align = channels * bits/sample / 8
    short bps2;        // 34 bits per sample
    char data[4];      // 36 "data"
    long dlength;      // 40 data length (bytes)
} waveheader;

//****************************************************************************

void sndfile::ReadWAV(const char* path)
{
    DataLocation = 44;
    waveheader* wave = (waveheader*)Hdrbuf;
    short type = GetShort((unsigned char*)Hdrbuf + 20);

    if (type != 1)
        if (type > 255)
            Swap = 1;
        else error("sndfile::ReadWAV", "All sound files must be type PCM.\n"
                   "Offending file:  \"%s\", format = %d", path, type);
    short bps2 = GetShort((unsigned char*)Hdrbuf + 34);
    if (bps2 != 16)
        error("sndfile::ReadWAV", "All sound files must be 16 bit.\n"
              "Offending file:  \"%s\"\n Bits per sample = %d",
              path, bps2);
    Channels = GetShort((unsigned char*)Hdrbuf + 22);
    Srate = int(GetLong((unsigned char*)Hdrbuf + 24));

    // CLM, for one, writes a comment where "data" should appear so we have to
    // search for "data" instead, or is CLM right and the spec wrong?
    if (strncmp(wave->data, "data", 4)) {
      // start where "data" should have been (byte # 36)
      char* data = bytesstr((char*)wave + 35, "data", HDRSIZE - 36);
      if (data == NULL)
        error("sndfile::ReadWAV", "'%s' does not appear to be a valid wave "
              "file", path);
      // I don't know why, but we always seem to think we have 8 bytes more
      // than we really have.  Perhaps CLM writes the wrong length?
      // wave->dlength = *((long*)(data + 4)) - 8;
      wave->dlength = GetLong((unsigned char*)data + 4) - 8;
      DataLocation = (data - (char*)wave) + 8;
    }
    NumFrames = wave->dlength / (2 * Channels);
}

//****************************************************************************

// Like strstr() only works with a byte stream (the first arg) that may contain
// '\0' chars or anything else.  <match> should be a normal string.

char* sndfile::bytesstr(char* strg, char* match, int maxchars) 
{
  int len = strlen(match);
  int i;
  int matched;
  char* ptr;
  char* result = NULL;

  for (i = 0, ptr = strg; i < maxchars; ++i, ++ptr) {
    if (*ptr == *match) {
      matched = 1;
      for (int j = 1; j < len; ++j)
        if (*(ptr + j) == *(match + j))
          ++matched;
      if (matched == len) {
        result = ptr;
        break;
      }
    }
  }
  return result;
}

//****************************************************************************
//
// Byte positions for ".aiff" type sound files:
// (Taken from Bill Schottsteadt's CLM/headers.c)
//
// 0: "FORM"
// 4: size (bytes)
// 8: "AIFF" or "AIFC" -- the latter includes compressed formats.
//
// Thereafter the file is organized into "chunks", each chunk being
// a 4-byte identifer followed by an int (4-bytes) giving the chunk size
// not including the 8-byte header.  AIFF data is signed.  If the chunk
// size is odd, an extra (unaccounted-for) null byte is added at the end.
//
// The chunks we want are "COMM" and "SSND".
//
// COMM: 0: chans
//       2: frames
//       6: bits per sample
//       8: srate as 80-bit IEEE float
// then if AIFC (not AIFF), 4 bytes giving compression id ("NONE"=not
// compressed) followed by Pascal string giving long name of compression
// type.
// SSND: 0: data location (offset within SSND chunk).
//
//*****************************************************************************

// Read an AIFF header. 

void sndfile::ReadAIFF(const char *path)
{
    // The sound file format.  Should be 16 (which is 16-bit linear). 
    int format;
    // Counts the number of bytes processed. 
    int count;
    // We're going to be incrementing this pointer and we don't want
    // to lose the <Hdrbuf> pointer because we need to free it (all). 
    char *ptr = Hdrbuf;

    // Make sure this is an AIFF file and not an AIFC compressed file. 
    if (strncmp(ptr + 8, "AIFF", 4) != 0)
        error("sndfile::ReadAIFF",
              "Not supporting AIFF compressed format.\n%s.",
              path);
    ptr += 12;                  // Skip the first block "FORMxxxxAIFF" 
    count = 12;                 // Ditto 
    
     // Skip along to the COMM block--we should already be there, but best to
     // be safe. 

    while (strncmp(ptr, "COMM", 4)) {
        count += 2;
        ptr += 2;
    }

    // format of 16 is 16-bit linear (same as format 3 in ".snd" files). 
    format = GetShort((unsigned char*)ptr + 14);
    if (format != 16)
        if (format == 4096) // = 16 byte swapped 
            Swap = 1;
        else error("sndfile::ReadAIFF", 
                   "All sound files must be 16-bit linear.\n"
                   "Offending file:  %s\nformat = %d", path, format);

    // Get the data from the correct byte positions as detailed above. 
    Channels = GetShort((unsigned char*)ptr + 8);
    NumFrames = int(GetLong((unsigned char*)ptr + 10));

    // Convert that horrible sampling rate double format. 
    Srate = (int)IEEE80ToDouble((unsigned char*)ptr + 16);

    // Skip along to the SSND block to get the <DataLocation>. According to
    // the specification, we should be able to get the chunk size and skip
    // that many bytes to read the next chunk header until we get to the SSND
    // block.  However, when we have header sizes above 200 bytes this
    // doesn't seem to work (can't find out why).  So we have to skip two
    // bytes at a time and see if we got to the SSND block the slow way. 
    while (strncmp(ptr, "SSND", 4)) {
        count += 2;
        
        // Don't generate pointer errors!  If we haven't got there by now,
        // something is wrong. 
        if (count > HDRSIZE) 
          error("sndfile::ReadAIFF", "Error reading header: %s", path);
        ptr += 2;
    }
    DataLocation = int(GetLong((unsigned char*)ptr + 8) + count + 16);
}

//****************************************************************************
//
// Byte positions for ".snd" type sound files:
// (Taken from Bill Schottsteadt's headers.c)
//
//   0:  ".snd"
//   4:  data_location (bytes) (not necessarily word aligned on Sun)
//   8:  data_size (bytes) -- sometimes incorrect ("advisory")
//   12: data format indicator -- see below
//   16: srate (int)
//   20: chans
//   24: comment start
//
// Formats are:
//
// 0 unspecified, 1 mulaw_8, 2 linear_8, 3 linear_16, 4 linear_24,
// 5 linear_32, 6 float, 7 double, 8 indirect, 9 nested, 10 dsp_core,
// 11 dsp_data_8, 12 dsp_data_16, 13 dsp_data_24, 14 dsp_data_32,
// 16 display, 17 mulaw_squelch, 18 emphasized, 19 compressed,
// 20 compressed_emphasized, 21 dsp_commands, 22 dsp_commands_samples,
// 23 adpcm_g721, 24 adpcm_g722, 25 adpcm_g723, 26 adpcm_g723_5,
// 27 alaw_8, 28 aes, 29 delat_mulaw_8.
//
//*****************************************************************************

// Read an SND header.  This is a lot less hassle than AIFF. 

void sndfile::ReadSND(const char *path)
{
    long format;

    DataLocation = int(GetLong((unsigned char*)Hdrbuf + 4));
    format = GetLong((unsigned char*)Hdrbuf + 12);
    if (format != 3)
        if (format == (768)) // the number 3, byte-swapped
            Swap = 1;
        else error("sndfile::ReadSND",
                   "All sound files must be 16-bit linear.\n"
                   "Offending file:  \"%s\"", path);
    Srate = int(GetLong((unsigned char*)Hdrbuf + 16));
    Channels = int(GetLong((unsigned char*)Hdrbuf + 20));
    
     // Data size is in bytes so do appropriate scaling to get the number of
     // frames. 
    NumFrames = int(GetLong((unsigned char*)Hdrbuf + 8)) / 
        (sizeof(short) * Channels);
}

//*****************************************************************************

// Get a 4-byte integer from the character pointer <start>.  Get it right
// whether this is little or big endian. 

inline long sndfile::GetLong(unsigned char *start)
{
    unsigned char c1 = *start;
    unsigned char c2 = *(start + 1);
    unsigned char c3 = *(start + 2);
    unsigned char c4 = *(start + 3);
    long l;

#ifdef IS_LITTLE_ENDIAN
    l = (((long)c4 << 24) |
         ((long)c3 << 16) |
         ((long)c2 << 8) |
         (long)c1);
#else
    l = (((long)c1 << 24) |
         ((long)c2 << 16) |
         ((long)c3 << 8) | 
         ((long)c4));
#endif

    if (Swap)
        SwapLong(&l);
    return l;
}

//*****************************************************************************

inline long sndfile::SwapLong(long* lp)
{
    unsigned char *p = (unsigned char *)lp;
    unsigned char c = p[0];
    p[0] = p[3];
    p[3] = c;
    c = p[1];
    p[1] = p[2];
    p[2] = c;
    return long(*p);
}

//*****************************************************************************

// Swap two bytes and return a short int.  Endianness-immune also.

inline short sndfile::GetShort(unsigned char *start)
{
    unsigned char c1 = *start;
    unsigned char c2 = *(start + 1);
    short s;

#ifdef IS_LITTLE_ENDIAN
    s = (((short)c2 << 8) |
         (short)c1);
#else
    s = (((short)c1 << 8) |
         (short)c2);
#endif

    if (Swap)
        myswab((char*)&s, 2);
    return s;
}

//*****************************************************************************

// Following two functions stolen wholesale from Bill Schottsteadt's
// headers.c.  Called by sndfile::ReadAIFF(). 

double sndfile::IEEE80ToDouble(unsigned char* p)
{
    char sign;
    short exp = 0;
    unsigned long mant1 = 0;
    unsigned long mant0 = 0;
    double val;

    exp = *p++;
    exp <<= 8;
    exp |= *p++;
    sign = (exp & 0x8000) ? 1 : 0;
    exp &= 0x7FFF;
    mant1 = *p++;
    mant1 <<= 8;
    mant1 |= *p++;
    mant1 <<= 8;
    mant1 |= *p++;
    mant1 <<= 8;
    mant1 |= *p++;
    mant0 = *p++;
    mant0 <<= 8;
    mant0 |= *p++;
    mant0 <<= 8;
    mant0 |= *p++;
    mant0 <<= 8;
    mant0 |= *p++;
    if (mant1 == 0 && mant0 == 0 && exp == 0 && sign == 0)
        return 0.0;
    else {
        val = myUlongToDouble(mant0) * pow(2.0, -63.0);
        val += myUlongToDouble(mant1) * pow(2.0, -31.0);
        val *= pow(2.0, ((double)exp) - 16383.0);
        return sign ? -val : val;
    }
}

//*****************************************************************************

// Called by sndfile::IEEE80ToDouble(). 

inline double sndfile::myUlongToDouble(unsigned long ul)
{
    const unsigned long ulpow2to31 = 0x80000000L;
    const double dpow2to31 = 2147483648.0;
    double val;

    if (ul & ulpow2to31)
        val = dpow2to31 + (ul & (~ulpow2to31));
    else
        val = ul;
    return val;
}

//*****************************************************************************

// Can't find which library to link against with stdlib's swab function, and
// the declaration in the header doesn't seem to work either so here's my own
// and basta.

inline void sndfile::myswab(char* buf, int len)
{
    char* ptr = buf;
    char c;

    for (int i = 0; i < len; i += 2, ptr += 2) {
        c = *ptr;
        *ptr = *(ptr + 1);
        *(ptr + 1) = c;
    }
}

//*****************************************************************************
//*****************************************************************************

// Simple error handling function.  Forces exit of programme. Works exactly
// like any of the printf family, except that it prints an introductory
// message supplying the function (<function_name>) that signalled the error
// as well as the user supplied message with all the usual formatting
// characteristics. 
 
void error(const char *function_name, const char *format,...)
{
    va_list args;

    va_start(args, format);
    fprintf(stderr, "*** Error signalled by function %s():\n\r",
            function_name);
    vfprintf(stderr, format, args);
    fprintf(stderr, "\n\r");
    va_end(args);
    exit(1);
}

//*****************************************************************************

// EOF sndfile.cpp
 
