//*****************************************************************************
//
// File:             sndfile.h
//
// Purpose:          Header file for sndfile class.
//
// Author:           Michael Edwards - m@michael-edwards.org
//
// Date:             December 19th 1995
//
// License:          Copyright (C) 2001 Michael Edwards
//
//                   This file is part of rmsps.  
//
//                   rmsps is free software; you can redistribute it and/or
//                   modify it under the terms of the GNU General Public
//                   License as published by the Free Software Foundation;
//                   either version 2 of the License, or (at your option) any
//                   later version.
//
//                   rmsps is distributed in the hope that it will be useful,
//                   but WITHOUT ANY WARRANTY; without even the implied
//                   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//                   PURPOSE.  See the GNU General Public License for more
//                   details.
//
//                   You should have received a copy of the GNU General Public
//                   License along with rmsps; if not, write to the Free
//                   Software Foundation, Inc., 59 Temple Place, Suite 330,
//                   Boston, MA 02111-1307 USA
//
// $$ Last modified: 13:32:57 Fri Oct 12 2001 W. Europe Daylight Time
//
//*****************************************************************************

#ifndef __SNDFILE_H__
#define __SNDFILE_H__

#ifdef WIN32
#define IS_LITTLE_ENDIAN
#endif

void myswab(char* buf, int len);
void error(const char *function_name, const char *format,...);

class sndfile {

public:

    // Constructors. 
    sndfile(const char *path);
    sndfile(void);

    // Destructor. 
    virtual ~sndfile(void);

    // open a sound file and parse its header etc.
    int open(const char* path);

    int isopen(void) const { return Fp ? 1 : 0; }

    // close the sound file
    void close(void);

    void seek(double secs);

    float* GetSamples(int num);

    // Accessor for <Channels>.
    int GetChannels(void) const { return Channels; }

    // Accessor for <Fp>.
    FILE *GetFp(void) const { return Fp; }

    int GetDataLocation(void) { return DataLocation; }

    // Number of sample **frames** (not samples) in the sound file. 
    int GetNumFrames(void) { return NumFrames; }

    int GetSrate(void) const { return Srate; }

protected:

    // The sampling rate of the sound.  Only 22.05 and 44.1 kHz are legal. 
    int Srate;

    // Where the samples actually start in the file. 
    int DataLocation;

    // Number of channels in the sound file. 
    int Channels;

    // Number of sample **frames** (not samples) in the sound file. 
    int NumFrames;

    // the full path of the sound file.
    char *Path;

    // The file pointer used to access the samples. 
    FILE *Fp;

private:

    // whether to swab bytes for endianness problems.
    int Swap;

    // The index into the sample buffer of GetSamples().
    int Index;

    // sample buffer, allocated as necessary.
    float* Samples;

    // how many Samples were allocated for.
    int NumSamples;

    void AllocSamples(int num);

    // Reader for AIFF files (Mac, SGI). 
    void ReadAIFF(const char *path);

    // Reader for SND files (NeXT, Sun). 
    void ReadSND(const char *path);

    // Reader for WAVE files (win95)
    void ReadWAV(const char *path);

    char* bytesstr(char* string, char* match, int maxchars);

    // Conversion function for the awkward sampling rate field of AIFF files.
    double IEEE80ToDouble(unsigned char* p);

    // Function used in IEEE80ToDouble(). 
    inline double myUlongToDouble(unsigned long ul);

    // Return a four-byte int from a char pointer. 
    inline long GetLong(unsigned char *start);

    // Return a two-byte int from a char pointer. 
    inline short GetShort(unsigned char *start);

    // replacement of swab() for byte swapping.
    inline void myswab(char* buf, int len);

    inline long SwapLong(long* lp);

    // Place to store HDRSIZE number of chars for reading the header. 
    char *Hdrbuf;

    // There is no copy constructor. 
    //sndfile(const sndfile &old_class);
};

#endif // __SNDFILE_H__

//*****************************************************************************

// EOF sndfile.h
